import { useState, useEffect } from 'react'
import { motion } from 'framer-motion'
import { Music2, Star, Calendar, User } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyClasses.css'

const MyClasses = () => {
  const [classes, setClasses] = useState([])

  useEffect(() => {
    loadClasses()
    const interval = setInterval(loadClasses, 500)
    return () => clearInterval(interval)
  }, [])

  const loadClasses = () => {
    const saved = localStorage.getItem('danceClasses')
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        setClasses(parsed)
      } catch (e) {
        setClasses([])
      }
    } else {
      const defaultClasses = [
        {
          id: 1,
          style: 'Хип-хоп',
          instructor: 'Алексей Воронов',
          date: '2025-03-28',
          rating: 5
        },
        {
          id: 2,
          style: 'Бачата',
          instructor: 'Мария Соколова',
          date: '2025-03-26',
          rating: 5
        },
        {
          id: 3,
          style: 'Зумба',
          instructor: 'Елена Петрова',
          date: '2025-03-24',
          rating: 4
        }
      ]
      setClasses(defaultClasses)
      localStorage.setItem('danceClasses', JSON.stringify(defaultClasses))
    }
  }

  const totalClasses = classes.length
  const uniqueInstructors = new Set(classes.map(c => c.instructor)).size
  const averageRating = classes.length > 0 
    ? (classes.reduce((sum, c) => sum + c.rating, 0) / classes.length).toFixed(1)
    : 0

  return (
    <div className="my-classes">
      <div className="container">
        <motion.section 
          className="hero-section"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5 }}
        >
          <h1 className="hero-title">Мои занятия</h1>
          <p className="hero-description">
            Ваш прогресс в танцевальных занятиях
          </p>
        </motion.section>

        <motion.div 
          className="stats-grid"
          initial={{ opacity: 0, y: 20 }}
          animate={{ opacity: 1, y: 0 }}
          transition={{ duration: 0.5, delay: 0.2 }}
        >
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, scale: 1.02, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper">
              <Music2 size={40} strokeWidth={2.5} className="stat-icon" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{totalClasses}</div>
              <div className="stat-label">Занятий</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, scale: 1.02, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper">
              <Star size={40} strokeWidth={2.5} className="stat-icon" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{averageRating}</div>
              <div className="stat-label">Средняя оценка</div>
            </div>
          </motion.div>
          <motion.div 
            className="stat-card"
            whileHover={{ y: -8, scale: 1.02, transition: { duration: 0.3 } }}
          >
            <div className="stat-icon-wrapper">
              <User size={40} strokeWidth={2.5} className="stat-icon" />
            </div>
            <div className="stat-content">
              <div className="stat-value">{uniqueInstructors}</div>
              <div className="stat-label">Преподавателей</div>
            </div>
          </motion.div>
        </motion.div>

        <ImageSlider images={[
          '/images/dance-1.jpg',
          '/images/dance-2.jpg',
          '/images/dance-3.jpg',
          '/images/dance-4.jpg'
        ]} />

        <motion.section 
          className="classes-section"
          initial={{ opacity: 0 }}
          animate={{ opacity: 1 }}
          transition={{ duration: 0.5, delay: 0.3 }}
        >
          <h2 className="section-title">Ваши занятия</h2>
          {classes.length > 0 ? (
            <div className="classes-list">
              {classes.map((classItem, index) => (
                <motion.div 
                  key={classItem.id} 
                  className="class-item"
                  initial={{ opacity: 0, x: -20 }}
                  animate={{ opacity: 1, x: 0 }}
                  transition={{ delay: index * 0.1, duration: 0.4 }}
                  whileHover={{ x: 4, transition: { duration: 0.2 } }}
                >
                  <div className="class-main">
                    <div className="class-info">
                      <div className="class-style">{classItem.style}</div>
                      <div className="class-details">
                        <div className="class-detail">
                          <User size={18} strokeWidth={2} />
                          <span>{classItem.instructor}</span>
                        </div>
                        <div className="class-detail">
                          <Calendar size={18} strokeWidth={2} />
                          <span>
                            {new Date(classItem.date).toLocaleDateString('ru-RU', { 
                              day: 'numeric', 
                              month: 'long', 
                              year: 'numeric' 
                            })}
                          </span>
                        </div>
                      </div>
                    </div>
                    <div className="class-rating">
                      {Array.from({ length: 5 }, (_, i) => (
                        <Star
                          key={i}
                          size={20}
                          fill={i < classItem.rating ? '#2C3E50' : 'none'}
                          stroke={i < classItem.rating ? '#2C3E50' : '#BDC3C7'}
                          strokeWidth={2}
                        />
                      ))}
                    </div>
                  </div>
                </motion.div>
              ))}
            </div>
          ) : (
            <div className="empty-state">
              <Music2 size={64} strokeWidth={1.5} />
              <h3 className="empty-title">Нет занятий</h3>
              <p className="empty-text">Запишитесь на занятие прямо сейчас</p>
            </div>
          )}
        </motion.section>
      </div>
    </div>
  )
}

export default MyClasses

